#!/usr/bin/env python
#
#    This is PNCconf, a graphical configuration editor for LinuxCNC
#    Copyright 2007 Jeff Epler <jepler@unpythonic.net>
#    Pncconf 1.1 revamped by Chris Morley 2014
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
#    This builds the HAL files from the collected data.
#
import os
import time
import shutil

class HAL:
    def __init__(self,app):
        # access to:
        self.d = app.d  # collected data
        global SIG
        SIG = app._p    # private data (signal names)
        self.a = app    # The parent, pncconf
        global _PD
        _PD = app._p    # private data





    def write_halfile(self, base):
        def writebackup(origname):
            path, ext = os.path.splitext(origname)
            name = path.replace(base+"/","")
            print base
            print path,name,ext
            shutil.copy(origname ,os.path.join(base,"backups",name + str(time.time()).replace('.', '') + ext) )
        self._substitution_list = []
        axis_convert = ("x","y","z","a")
        halui_cmd_count = 0
        filename = os.path.join(base, self.d.machinename + ".hal")
        file = open(filename, "w")
        print >>file, _("# Generated by PNCconf at %s") % time.asctime()
        print >>file, _("# Using LinuxCNC version:  %s") % self.d.linuxcnc_version
        print >>file, _("# If you make changes to this file, they will be")
        print >>file, _("# overwritten when you run PNCconf again")
        print >>file
        if self.d.pyvcp and not self.d.frontend == _PD._AXIS:
            size = pos = geo = ""
            if self.d.pyvcpposition or self.d.pyvcpsize:
                if self.d.pyvcpposition:
                    pos = "+%d+%d"% (self.d.pyvcpxpos,self.d.pyvcpypos)
                if self.d.pyvcpsize:
                    size = "%dx%d"% (self.d.pyvcpwidth,self.d.pyvcpheight)
                geo = " -g %s%s"%(size,pos)
            print >>file, "loadusr -Wn pyvcp pyvcp%s -c pyvcp [DISPLAY](PYVCP)"%(geo)
            print >>file, "source postgui_call_list.hal"
        if self.d.gladevcp and self.d.standalonegvcp:
            fmax = geo = pos = size =  ""
            if self.d.gladevcpposition or self.d.gladevcpsize:
                if self.d.gladevcpposition:
                    pos = "+%d+%d"% (self.d.gladevcpxpos,self.d.gladevcpypos)
                if self.d.gladevcpsize:
                    size = "%dx%d"% (self.d.gladevcpwidth,self.d.gladevcpheight)
                geo = " -g %s%s"%(size,pos)
            if self.d.gladevcpforcemax: fmax = " -m True"
            theme = self.d.gladevcptheme
            if theme == "Follow System Theme":theme = ""
            else: theme = " -t "+theme
            print >>file, "loadusr -Wn gladevcp gladevcp -c gladevcp%s%s%s -H gvcp_call_list.hal gvcp-panel.ui"%(theme,fmax,geo)
        print >>file, "loadrt [KINS]KINEMATICS"
        print >>file, "loadrt [EMCMOT]EMCMOT servo_period_nsec=[EMCMOT]SERVO_PERIOD num_joints=[KINS]JOINTS"
        # pre process mesa commands
        mesa_load_cmnd,mesa_read_cmnd,mesa_write_cmnd = self.a.hostmot2_command_string(substitution = self.d.useinisubstitution)
        if self.d.number_pports:
            # pre process pport commands
            pport_load_cmnd,pport_read_cmnd,pport_write_cmnd = self.a.pport_command_string()
        # print the mesa load commands
        for i in mesa_load_cmnd:
            print >>file, "%s"%i

        if self.d.number_pports>0:
            # print the pport load commands
            for i in pport_load_cmnd:
                print >>file, "%s"%i


        if self.d.joystickjog:
            print >>file, "loadusr -W hal_input -KRAL %s\n"% self.d.usbdevicename

        spindle_enc = counter = probe = pwm = pump = estop = False 
        spindle_on = spindle_cw = spindle_ccw = False
        mist = flood = brake = at_speed = bldc = False

        if self.a.findsignal("s-encoder-a"):
            spindle_enc = True        
        if self.a.findsignal("probe"):
            probe = True
        if self.a.findsignal("s-pwm-pulse"):
            pwm = True
        if self.a.findsignal("charge-pump"):
            pump = True
        if self.a.findsignal("estop-ext"):
            estop = True
        if self.a.findsignal("spindle-enable"):
            spindle_on = True
        if self.a.findsignal("spindle-cw"):
            spindle_cw = True
        if self.a.findsignal("spindle-ccw"):
            spindle_ccw = True
        if self.a.findsignal("coolant-mist"):
            mist = True
        if self.a.findsignal("coolant-flood"):
            flood = True
        if self.a.findsignal("spindle-brake"):
            brake = True
        if self.a.findsignal("spindle-at-speed"):
            at_speed = True
        for i in self.d.available_axes:
            if self.d[i+"bldc_option"]:
                bldc = True
                break
        chargepump = self.a.findsignal("charge-pump-out")
        # load PID compnent:
        # if axis needs PID- (has pwm signal) then add its letter to pidlist
        temp = ""
        for i in self.d.available_axes:
            #print "looking at available axis : ",i
            #if not self.a.findsignal(i+"-encoder-a") and not self.a.findsignal(i+"-resolver"):
            #    continue
            temp = temp + "pid.%s,"%i
            tandemjoint = self.a.tandem_check(i)
            if tandemjoint:
                temp = temp + "pid.%s2,"%i
        # if user requested PID components add them to the list as well, starting at 0 and working up
        for i in range(0,self.d.userneededpid):
                temp=temp+"pid.%d,"% (i)
        pidlist = temp.rstrip(",")
        if not pidlist == "":
            print >>file, "loadrt pid names=%s"% pidlist

        if bldc or self.d.userneededbldc:
            self.d._bldcconfigstring = ""
            if bldc:
                for i in self.d.available_axes:
                    temp = self.d[i+"bldc_config"]
                    #print i,temp
                    if temp:
                        self.d._bldcconfigstring = self.d._bldcconfigstring + temp + ","
            if self.d.userneededbldc:
                    self.d._bldcconfigstring = self.d._bldcconfigstring + self.d.userneededbldc + ","
            temp = self.d._bldcconfigstring.rstrip(",")
            if not temp == "":
                print >>file, "loadrt bldc cfg=%s"% temp

        if (self.d.pyvcp and self.d.pyvcpconnect == 1) or self.d.gladevcp or self.d.userneededabs >0:
            self.d.absnames=""
            needed = False
            if self.d.gladevcp and self.d.spindlespeedbar: needed = True
            if self.d.pyvcphaltype == 1 and self.d.pyvcpconnect == 1 and self.d.pyvcp: needed = True
            if needed:
                self.d.absnames=self.d.absnames+"abs.spindle"
                if self.d.userneededabs >0:
                    self.d.absnames=self.d.absnames+","
            for i in range(0,self.d.userneededabs):
                self.d.absnames = self.d.absnames+"abs.%d"% (i)
                if i <> self.d.userneededabs-1:
                    self.d.absnames = self.d.absnames+","
            if not self.d.absnames == "":
                print >>file, "loadrt abs names=%s"% self.d.absnames

        if (self.d.pyvcp and self.d.pyvcpconnect == 1) or self.d.gladevcp or self.d.userneededlowpass >0:
            self.d.lowpassnames=""
            needed = False
            for i in range(0,self.d.userneededlowpass):
                self.d.lowpassnames = self.d.lowpassnames+"lowpass.%d,"% (i)
            if self.d.pyvcphaltype == 1 and self.d.pyvcpconnect == 1 and self.d.pyvcp: needed = True
            if self.d.gladevcp and self.d.spindlespeedbar: needed = True
            if needed:
                self.d.lowpassnames=self.d.lowpassnames+"lowpass.spindle"
            temp = self.d.lowpassnames.rstrip(",")
            self.d.lowpassnames = temp
            if not self.d.lowpassnames == "":
                print >>file, "loadrt lowpass names=%s"% temp

        pytest = self.d.pyvcp and self.d.pyvcphaltype == 1 and self.d.pyvcpconnect == 1
        gladetest = self.d.gladevcp and self.d.spindlespeedbar
        self.d.scalenames=[]
        names=''
        if spindle_enc and (pytest or gladetest):
            self.d.scalenames.append("scale.spindle")
        if self.d.suseoutputrange2:
            self.d.scalenames.append("scale.gear")
        if self.d.userneededscale >0:
            for i in range(0,self.d.userneededscale):
                self.d.scalenames.append("scale.%d"% (i))
        if not self.d.scalenames == []:
            for num,temp in enumerate(self.d.scalenames):
                if num ==0:
                    names+='%s'%temp
                else:
                    names+=',%s'%temp
            print >>file, "loadrt scale names=%s"% names
        if pump:
            print >>file, "loadrt charge_pump"
        if not at_speed and self.d.suseatspeed:
            print >>file, "loadrt near"
        if self.d.classicladder:
            print >>file, ("loadrt classicladder_rt numPhysInputs=%d numPhysOutputs=%d numS32in=%d"
                          " numS32out=%d numFloatIn=%d numFloatOut=%d numBits=%d numWords=%d") \
                          %(self.d.digitsin , self.d.digitsout , self.d.s32in, self.d.s32out, self.d.floatsin, self.d.floatsout,self.d.bitmem,self.d.wordmem)
        
        # load mux16
        self.d.mux16names=""
        for i in range(0,self.d.userneededmux16):
            self.d.mux16names = self.d.mux16names+"mux16.%d,"% (i)
        if self.d.joystickjog: 
           self.d.mux16names = self.d.mux16names+"jogspeed,"
        if self.d.externalmpg: 
            self.d.mux16names = self.d.mux16names+"jogincr,"  
        if self.d.externalfo: 
            self.d.mux16names = self.d.mux16names+"foincr,"
        if self.d.externalmvo: 
            self.d.mux16names = self.d.mux16names+"mvoincr,"
        if self.d.externalso: 
            self.d.mux16names = self.d.mux16names+"soincr,"
        if self.d.scaleselect: 
            self.d.mux16names = self.d.mux16names+"ratio_select,"
        temp = self.d.mux16names.rstrip(",")
        self.d.mux16names = temp
        if temp:
            print >>file, "loadrt mux16 names=%s"% (self.d.mux16names)

        # load user custom components
        for i in self.d.loadcompbase:
            if i == '': continue
            else:              
                print >>file, i 
        for i in self.d.loadcompservo:
            if i == '': continue
            else:              
                print >>file, i 

        print >>file
        # print parport read commands
        if self.d.number_pports:
            for i in pport_read_cmnd:
                print >>file, "%s"%i
 
        if pump: print >>file, "addf charge-pump servo-thread"
           
        for i in self.d.addcompbase:
            if not i == '':
                print >>file, i +" base-thread"

        # print mesa read commands
        if self.d.number_mesa:
            for i in mesa_read_cmnd:
                print >>file, "%s"%i
            
        print >>file, "addf motion-command-handler   servo-thread"
        print >>file, "addf motion-controller        servo-thread"

        if not pidlist == "":
            temp = pidlist.split(",")
            for i in temp:
                print >>file, "addf %s.do-pid-calcs       servo-thread"% i
        
        if bldc or self.d.userneededbldc:
            temp=self.d._bldcconfigstring.split(",")
            for num,j in enumerate(temp):
                print >>file, "addf bldc.%d servo-thread"% num

        if self.d.classicladder:
            print >>file,"addf classicladder.0.refresh servo-thread"

        # mux16 addf 
        temp=self.d.mux16names.split(",")
        if not temp == ['']:
            for j in (temp):
                j ='{0:<24}'.format(j)
                print >>file, "addf %s servo-thread"% j
        # scale addf
        needed = False
        if not self.d.scalenames == '':
            for j in self.d.scalenames:
                j ='{0:<24}'.format(j)
                print >>file, "addf %s servo-thread"% j
        # abs addf
        needed = False
        if self.d.pyvcp and self.d.pyvcphaltype == 1 and self.d.pyvcpconnect == 1: needed = True
        if self.d.userneededabs > 0 or (self.d.gladevcp and self.d.spindlespeedbar): needed = True
        if needed:
            temp=self.d.absnames.split(",")
            for j in (temp):
                j ='{0:<24}'.format(j)
                print >>file, "addf %s servo-thread"% j
        # lowpass addf
        needed = False
        if self.d.pyvcp and self.d.pyvcphaltype == 1 and self.d.pyvcpconnect == 1: needed = True
        if self.d.userneededlowpass > 0 or (self.d.gladevcp and self.d.spindlespeedbar): needed = True
        if needed:
            temp=self.d.lowpassnames.split(",")
            for j in (temp):
                j ='{0:<24}'.format(j)
                print >>file, "addf %s servo-thread"% j

        for i in self.d.addcompservo:
            if not i == '':
                print >>file, i +" servo-thread"
        if not at_speed and self.d.suseatspeed:
            print >>file, "addf near.0                   servo-thread"
        # print parport write commands
        if self.d.number_pports:
            for i in pport_write_cmnd:
                print >>file, "%s"%i
        if self.d.number_mesa:
            # print mesa write commands
            for i in mesa_write_cmnd:
                print >>file, "%s"%i

        if chargepump:
            steppinname = self.a.make_pinname(chargepump, substitution = self.d.useinisubstitution)
            print >>file
            print >>file, "# ---Chargepump StepGen: 0.25 velocity = 10Khz square wave output---"
            print >>file
            print >>file, "setp   " + steppinname + ".dirsetup        100"
            print >>file, "setp   " + steppinname + ".dirhold         100"
            print >>file, "setp   " + steppinname + ".steplen         100"
            print >>file, "setp   " + steppinname + ".stepspace       100"
            print >>file, "setp   " + steppinname + ".position-scale  10000"
            print >>file, "setp   " + steppinname + ".step_type       2"
            print >>file, "setp   " + steppinname + ".control-type    1"
            print >>file, "setp   " + steppinname + ".maxaccel        0"
            print >>file, "setp   " + steppinname + ".maxvel          0"
            print >>file, "setp   " + steppinname + ".velocity-cmd    0.25"
            print >>file
            print >>file, "net x-enable                                 => " + steppinname +".enable"

        print >>file
        self.connect_output(file)              
        print >>file
        self.connect_input(file)
        print >>file

        ##############################################################
        # connect joints 
        ##############################################################
        # self.d.axes:
        # 0 = xyz
        # 1 = xz
        # 2 = xyza

        if not self.d.axes in(0,1,2):
            print 'error in number of axis identity: ', self.d.axes
            return
        jnum = 0
        # Always add X axis
        self.connect_joint(file, jnum, 'x')
        tandemjoint = self.a.tandem_check('x')
        if tandemjoint:
            jnum += 1
            self.connect_joint(file, jnum, 'x2')

        # Maybe add Y Axis ###################
        if self.d.axes in(0,1):
            jnum += 1
            self.connect_joint(file, jnum, 'y')
            tandemjoint = self.a.tandem_check('y')
            if tandemjoint:
                jnum += 1
                self.connect_joint(file, jnum, 'y2')

        # Always add Z Axis ##################
        jnum += 1
        self.connect_joint(file, jnum, 'z')
        tandemjoint = self.a.tandem_check('z')
        if tandemjoint:
            jnum += 1
            self.connect_joint(file, jnum, 'z2')

        # Maybe add A axis ###################
        if self.d.axes == 1:
            jnum += 1
            self.connect_joint(file, jnum, 'a')
            tandemjoint = self.a.tandem_check('a')
            if tandemjoint:
                jnum += 1
                self.connect_joint(file, jnum, 'a2')

        # usually add Spindle ##################
        if self.d.include_spindle:
            self.connect_joint(file, 0, 's') # 9 for [SPINDLE_9]

        print >>file
        print >>file, "#******************************"
        print >>file, _("# connect miscellaneous signals") 
        print >>file, "#******************************"
        print >>file
        print >>file, _("#  ---HALUI signals---")
        print >>file

        jnum = 0
        for axletter in axis_convert:
            if axletter in self.d.available_axes:
                # support for KINEMATICS_IDENTITY kins only 
                # Assumption: gui uses halui teleop jogging for KINEMATICS_IDENTITY configs
                #             (axis gui does this for joints_axes)
                print >>file, "net axis-select-%s  halui.axis.%s.select"% (axletter,axletter)
                print >>file, "net jog-%s-pos      halui.axis.%s.plus"% (axletter,axletter)
                print >>file, "net jog-%s-neg      halui.axis.%s.minus"% (axletter,axletter)
                print >>file, "net jog-%s-analog   halui.axis.%s.analog"% (axletter,axletter)

                # joints only items (no corresponding axis item):
                print >>file, "net %s-is-homed     halui.joint.%d.is-homed"% (axletter,jnum)
                jnum = jnum + 1 # expect joints in sequence (like trivkins)

        print >>file, "net jog-selected-pos      halui.axis.selected.plus"
        print >>file, "net jog-selected-neg      halui.axis.selected.minus"
        print >>file, "net spindle-manual-cw     halui.spindle.0.forward"
        print >>file, "net spindle-manual-ccw    halui.spindle.0.reverse"
        print >>file, "net spindle-manual-stop   halui.spindle.0.stop"
        print >>file, "net machine-is-on         halui.machine.is-on"
        print >>file, "net jog-speed             halui.axis.jog-speed"
        print >>file, "net MDI-mode              halui.mode.is-mdi"
        print >>file
        if pump:    
            print >>file, _("#  ---charge pump signals---")
            print >>file, "net estop-out       =>  charge-pump.enable"
            print >>file, "net charge-pump     <=  charge-pump.out"
            print >>file
        print >>file, _("#  ---coolant signals---")
        print >>file
        print >>file, "net coolant-mist      <=  iocontrol.0.coolant-mist"
        print >>file, "net coolant-flood     <=  iocontrol.0.coolant-flood"
        print >>file
        print >>file, _("#  ---probe signal---")
        print >>file
        print >>file, "net probe-in     =>  motion.probe-input"
        print >>file
        if self.d.externaljog:
            print >>file, _("# ---jog button signals---")
            print >>file
            print >>file, "sets    jog-speed %f"% self.d.jograpidrate
            print >>file
        if self.d.joystickjog:
            print >>file, _("# ---USB device jog button signals---")
            print >>file
            print >>file, "# connect selectable mpg jog speeds "
            print >>file, "net jog-speed-a           =>  jogspeed.sel0"
            print >>file, "net jog-speed-b           =>  jogspeed.sel1"
            print >>file, "net jog-speed             <=  jogspeed.out-f"
            print >>file, "setp    jogspeed.in00          %f"% (self.d.joystickjograpidrate0)
            print >>file, "setp    jogspeed.in01          %f"% (self.d.joystickjograpidrate1)
            print >>file, "setp    jogspeed.in02          %f"% (self.d.joystickjograpidrate2)
            print >>file, "setp    jogspeed.in03          %f"% (self.d.joystickjograpidrate3)
            if not self.d.joycmdrapida =="":
                print >>file, "net jog-speed-a           <=  %s"% (self.d.joycmdrapida)
            if not self.d.joycmdrapidb =="":
                print >>file, "net jog-speed-b           <=  %s"% (self.d.joycmdrapidb)
            for axletter in axis_convert:
                if axletter in self.d.available_axes:
                    pin_pos = self.d["joycmd"+axletter+"pos"]
                    pin_neg = self.d["joycmd"+axletter+"neg"]
                    if not pin_pos == "" and not pin_neg =="":
                        print >>file, "net jog-%s-pos            %s"% (axletter,pin_pos)
                        print >>file, "net jog-%s-neg            %s"% (axletter,pin_neg)
                    pin_analog = self.d["joycmdanalog"+axletter]
                    if not pin_analog == "":
                        print >>file, "net jog-%s-analog         %s"% (axletter,pin_analog)
            print >>file

        # check for shared MPG 
        pinname = self.a.make_pinname(self.a.findsignal("select-mpg-a"), substitution = self.d.useinisubstitution)
        if pinname:
            print "shared MPG", pinname
            ending = ""
            if "enc" in pinname: ending = ".count"
            print >>file, "# ---jogwheel signals to mesa encoder - shared MPG---"
            print >>file
            print >>file, "net axis-selected-count     <=  %s%s"% (pinname,ending)
            if 'encoder' in ending:
                print >>file, "setp    %s.filter true" % pinname
                print >>file, "setp    %s.counter-mode true" % pinname
            print >>file
            # was jogging MPG option selected?
            if self.d.externalmpg:
                    print >>file, _("#  ---mpg signals---")
                    print >>file
                    if not self.d.multimpg:
                        for axletter in axis_convert:
                            if axletter in self.d.available_axes:
                                print >>file, "#       for axis %s MPG" % axletter
                                print >>file, "setp    axis.%s.jog-vel-mode 0" % axletter
                                print >>file, "net selected-jog-incr    =>  axis.%s.jog-scale" % axletter
                                print >>file, "net axis-select-%s       =>  axis.%s.jog-enable"% (axletter,axletter)
                                print >>file, "net axis-selected-count =>  axis.%s.jog-counts"% axletter
                            print >>file
        # check for dedicated axis MPG jogging option
        for axletter in axis_convert:
            if axletter in self.d.available_axes:
                pinname = self.a.make_pinname(self.a.findsignal(axletter+"-mpg-a"), substitution = self.d.useinisubstitution)
                if pinname:
                    ending = ""
                    if "enc" in pinname: ending = ".count"
                    print >>file, "# ---jogwheel signals to mesa encoder - %s axis MPG---"% axletter
                    print >>file
                    print >>file, "net %s-jog-count          <=  %s%s"% (axletter, pinname,ending)
                    if 'encoder' in ending:
                        print >>file, "setp    %s.filter true" % pinname
                        print >>file, "setp    %s.counter-mode false" % pinname
                    print >>file
                    if self.d.externalmpg:
                        print >>file, _("#  ---mpg signals---")
                        print >>file
                        if self.d.multimpg: # means MPG per axis
                            print >>file, "setp    axis.%s.jog-vel-mode 0" % axletter
                            print >>file, "net %s-jog-enable         =>  axis.%s.jog-enable"% (axletter, axletter)
                            print >>file, "net %s-jog-count          =>  axis.%s.jog-counts" % (axletter, axletter)
                            print >>file, "net selected-jog-incr    =>  axis.%s.jog-scale" % axletter
                            print >>file, "sets %s-jog-enable    true"% (axletter)
                            print >>file
        if self.d.externalmpg and not self.d.frontend == _PD._TOUCHY:# TOUCHY GUI sets its own jog increments:
            if self.d.incrselect :
                print >>file, "# connect selectable mpg jog increments "
                print >>file
                print >>file, "net jog-incr-a           =>  jogincr.sel0"
                print >>file, "net jog-incr-b           =>  jogincr.sel1"
                print >>file, "net jog-incr-c           =>  jogincr.sel2"
                print >>file, "net jog-incr-d           =>  jogincr.sel3"
                print >>file, "net selected-jog-incr    <=  jogincr.out-f"
                if self.d.mpgdebounce:
                    print >>file, "    setp jogincr.debounce-time      %f"% self.d.mpgdebouncetime
                print >>file, "    setp jogincr.use-graycode      %s"% self.d.mpggraycode
                print >>file, "    setp jogincr.suppress-no-input %s" % self.d.mpgignorefalse
                for i in range(0,16):
                    value = self.d["mpgincrvalue%d"% i]
                    print >>file, "    setp jogincr.in%02d          %f"% (i,value)
                print >>file
            else:
                print >>file, "sets selected-jog-incr     %f"% (self.d.mpgincrvalue0)
                print >>file

        # check for dedicated feed override MPG
        pinname = self.a.make_pinname(self.a.findsignal("fo-mpg-a"), substitution = self.d.useinisubstitution)
        if pinname:
            ending = ""
            if "enc" in pinname: ending = ".count"
            print >>file, "# ---feed override signals to mesa encoder - mpg---"
            print >>file
            print >>file, "net fo-count     <=  %s%s"% (pinname,ending)
            if 'encoder' in ending:
                print >>file, "setp    %s.filter true" % pinname
                print >>file, "setp    %s.counter-mode true" % pinname
            print >>file
        # was feed overrride option selected? MPG or switch selcted?
        if self.d.externalfo:
            if self.d.fo_usempg:
                print >>file, "# connect feed overide increments - MPG"
                print >>file
                print >>file, "    setp halui.feed-override.direct-value false"
                print >>file, "    setp halui.feed-override.scale .01"
                if pinname: # dedicated MPG
                    if self.a.findsignal("fo-enable"): # make it enable-able externally 
                        print >>file, "net  fo-enable           => halui.feed-override.count-enable"
                    else:
                        print >>file, "    setp halui.feed-override.count-enable true"
                    print >>file, "net fo-count            =>  halui.feed-override.counts"
                else: # shared MPG
                    print >>file, "net fo-enable            => halui.feed-override.count-enable"
                    print >>file, "net axis-selected-count => halui.feed-override.counts"
                print >>file
            elif self.d.fo_useswitch:
                print >>file, "# connect feed overide increments - switches"
                print >>file
                print >>file, "    setp halui.feed-override.count-enable true"
                print >>file, "    setp halui.feed-override.direct-value true"
                print >>file, "    setp halui.feed-override.scale .01"
                print >>file, "net feedoverride-incr   =>  halui.feed-override.counts"
                print >>file, "net fo-incr-a           =>  foincr.sel0"
                print >>file, "net fo-incr-b           =>  foincr.sel1"
                print >>file, "net fo-incr-c           =>  foincr.sel2"
                print >>file, "net fo-incr-d           =>  foincr.sel3"
                print >>file, "net feedoverride-incr   <=  foincr.out-s"
                if self.d.fodebounce:
                    print >>file, "    setp foincr.debounce-time      %f"% self.d.fodebouncetime
                print >>file, "    setp foincr.use-graycode      %s"% self.d.fograycode
                print >>file, "    setp foincr.suppress-no-input %s" % self.d.foignorefalse
                for i in range(0,16):
                    value = self.d["foincrvalue%d"% i]
                    print >>file, "    setp foincr.in%02d          %f"% (i,value)
                print >>file

        # check for dedicated max velocity MPG
        pinname = self.a.make_pinname(self.a.findsignal("mvo-mpg-a"), substitution = self.d.useinisubstitution)
        if pinname:
            ending = ""
            if "enc" in pinname: ending = ".count"
            print >>file, "# ---max velocity override signals to mesa encoder - mpg---"
            print >>file
            print >>file, "net mvo-count     <=  %s%s"% (pinname,ending)
            if 'encoder' in ending:
                print >>file, "setp    %s.filter true" % pinname
                print >>file, "setp    %s.counter-mode true" % pinname
            print >>file
        # was max velocity override option selected? MPG or switch selected?
        if self.d.externalmvo:
            temp=[]
            for i in self.d.available_axes:
                temp.append(float(self.d[i+"maxvel"]))
            scale = max(temp)/100
            if self.d.mvo_usempg:
                print >>file, "# connect max velocity overide increments - MPG"
                print >>file
                print >>file, "    setp halui.max-velocity.direct-value false"
                print >>file, "    setp halui.max-velocity.scale %04f"% scale
                if pinname: # dedicated MPG
                    if self.a.findsignal("mvo-enable"): # make it enable-able externally 
                        print >>file, "net mvo-enable           =>  halui.max-velocity.count-enable"
                    else:
                        print >>file, "    setp halui.max-velocity.count-enable true"
                    print >>file, "net mvo-count            =>  halui.max-velocity.counts"
                else: # shared MPG
                    print >>file, "net mvo-enable           =>  halui.max-velocity.count-enable"
                    print >>file, "net axis-selected-count =>  halui.max-velocity.counts"
                print >>file
            elif self.d.mvo_useswitch:
                print >>file, "# connect max velocity overide increments - switches"
                print >>file
                print >>file, "    setp halui.max-velocity.count-enable true"
                print >>file, "    setp halui.max-velocity.direct-value true"
                print >>file, "    setp halui.max-velocity.scale %04f"% scale
                print >>file, "net max-vel-override-incr   =>  halui.max-velocity.counts"
                print >>file, "net mvo-incr-a           =>  mvoincr.sel0"
                print >>file, "net mvo-incr-b           =>  mvoincr.sel1"
                print >>file, "net mvo-incr-c           =>  mvoincr.sel2"
                print >>file, "net mvo-incr-d           =>  mvoincr.sel3"
                print >>file, "net max-vel-override-incr   <=  mvoincr.out-s"
                if self.d.mvodebounce:
                    print >>file, "    setp mvoincr.debounce-time      %f"% self.d.mvodebouncetime
                print >>file, "    setp mvoincr.use-graycode      %s"% self.d.mvograycode
                print >>file, "    setp mvoincr.suppress-no-input %s" % self.d.mvoignorefalse
                for i in range(0,16):
                    value = self.d["mvoincrvalue%d"% i]
                    print >>file, "    setp mvoincr.in%02d          %f"% (i,value)
                print >>file

        # check for dedicated spindle override MPG
        pinname = self.a.make_pinname(self.a.findsignal("so-mpg-a"), substitution = self.d.useinisubstitution)
        if pinname:
            ending = ""
            if "enc" in pinname: ending = ".count"
            print >>file, "# ---spindle override signals to mesa encoder - mpg---"
            print >>file
            print >>file, "net so-count     <=  %s%s"% (pinname,ending)
            if 'encoder' in ending:
                print >>file, "setp    %s.filter true" % pinname
                print >>file, "setp    %s.counter-mode true" % pinname
            print >>file
        if self.d.externalso:
            if self.d.so_usempg:
                print >>file, "# connect spindle overide increments - MPG"
                print >>file
                print >>file, "    setp halui.spindle.0.override.direct-value false"
                print >>file, "    setp halui.spindle.0.override.scale .01"
                if pinname: # dedicated MPG
                    if self.a.findsignal("so-enable"): # make it enable-able externally
                        print >>file, "net so-enable             =>  halui.spindle.0.override.count-enable"
                    else:
                        print >>file, "    setp halui.spindle.0.override.count-enable true"
                    print >>file, "net so-count              =>  halui.spindle.0.override.counts"
                else: # shared MPG
                    print >>file, "net so-enable             =>  halui.spindle.0.override.count-enable"
                    print >>file, "net axis-selected-count  =>  halui.spindle.0.override.counts"
                print >>file
            elif self.d.so_useswitch:
                print >>file, "# connect spindle overide increments "
                print >>file
                print >>file, "    setp halui.spindle.0.override.count-enable true"
                print >>file, "    setp halui.spindle.0.override.direct-value true"
                print >>file, "    setp halui.spindle.0.override.scale .01"
                print >>file, "net spindleoverride-incr  =>  halui.spindle.0.override.counts"
                print >>file, "net so-incr-a             =>  soincr.sel0"
                print >>file, "net so-incr-b             =>  soincr.sel1"
                print >>file, "net so-incr-c             =>  soincr.sel2"
                print >>file, "net so-incr-d             =>  soincr.sel3"
                print >>file, "net spindleoverride-incr  <=  soincr.out-s"
                if self.d.sodebounce:
                    print >>file, "    setp soincr.debounce-time      %f"% self.d.sodebouncetime
                print >>file, "    setp soincr.use-graycode      %s"% self.d.sograycode
                print >>file, "    setp soincr.suppress-no-input %s" % self.d.soignorefalse
                for i in range(0,16):
                    value = self.d["soincrvalue%d"% i]
                    print >>file, "    setp soincr.in%02d          %f"% (i,value)
                print >>file
        print >>file, _("#  ---motion control signals---")
        print >>file
        print >>file, "net in-position               <=  motion.in-position"
        print >>file, "net machine-is-enabled        <=  motion.motion-enabled"
        print >>file
        print >>file, _("#  ---digital in / out signals---")
        print >>file
        for i in range(4):
            dout = "dout-%02d" % i
            if self.a.findsignal(dout):
                print >>file, "net %s     <=  motion.digital-out-%02d" % (dout, i)
        for i in range(4):
            din = "din-%02d" % i
            if self.a.findsignal(din):
                print >>file, "net %s     =>  motion.digital-in-%02d" % (din, i)
        print >>file, _("#  ---estop signals---")
        print >>file
        print >>file, "net estop-out     <=  iocontrol.0.user-enable-out"
        if  self.d.classicladder and self.d.ladderhaltype == 1 and self.d.ladderconnect: # external estop program
            print >>file
            print >>file, _("# **** Setup for external estop ladder program -START ****")
            print >>file
            print >>file, "net estop-out     => classicladder.0.in-00"
            print >>file, "net estop-ext     => classicladder.0.in-01"
            print >>file, "net estop-strobe     classicladder.0.in-02   <=  iocontrol.0.user-request-enable"
            print >>file, "net estop-outcl      classicladder.0.out-00  =>  iocontrol.0.emc-enable-in"
            print >>file
            print >>file, _("# **** Setup for external estop ladder program -END ****")
        elif estop:
            print >>file, "net estop-ext     =>  iocontrol.0.emc-enable-in"
        else:
            print >>file, "net estop-out     =>  iocontrol.0.emc-enable-in"
        print >>file
        if self.d.toolchangeprompt:
            print >>file, _("#  ---manual tool change signals---")
            print >>file
            print >>file, "loadusr -W hal_manualtoolchange"
            print >>file, "net tool-change-request     iocontrol.0.tool-change       =>  hal_manualtoolchange.change"
            print >>file, "net tool-change-confirmed   iocontrol.0.tool-changed      <=  hal_manualtoolchange.changed"
            print >>file, "net tool-number             iocontrol.0.tool-prep-number  =>  hal_manualtoolchange.number"
            print >>file, "net tool-prepare-loopback   iocontrol.0.tool-prepare      =>  iocontrol.0.tool-prepared"
            print >>file
        else:
            print >>file, _("#  ---toolchange signals for custom tool changer---")
            print >>file
            print >>file, "net tool-number             <=  iocontrol.0.tool-prep-number"
            print >>file, "net tool-change-request     <=  iocontrol.0.tool-change"
            print >>file, "net tool-change-confirmed   =>  iocontrol.0.tool-changed" 
            print >>file, "net tool-prepare-request    <=  iocontrol.0.tool-prepare"
            print >>file, "net tool-prepare-confirmed  =>  iocontrol.0.tool-prepared" 
            print >>file
        if self.d.classicladder:
            print >>file
            if self.d.modbus:
                print >>file, _("# Load Classicladder with modbus master included (GUI must run for Modbus)")
                print >>file
                print >>file, "loadusr classicladder --modmaster custom.clp"
                print >>file
            else:
                print >>file, _("# Load Classicladder without GUI (can reload LADDER GUI in AXIS GUI")
                print >>file
                print >>file, "loadusr classicladder --nogui custom.clp"
                print >>file
            if self.d.laddertouchz:
                othercmds = self.d.gladevcphaluicmds
                print >>file, _("#  --- Classicladder signals for Z axis Auto touch off program---")
                print >>file, "net auto-touch-z    =>   classicladder.0.in-00"
                print >>file, "net MDI-mode        =>   classicladder.0.in-01"
                print >>file, "net in-position     =>   classicladder.0.in-02"
                print >>file, "net z-touchoff-cmd       classicladder.0.out-00   =>    halui.mdi-command-%02d"% (othercmds)
                print >>file, "net z-zero-cmd           classicladder.0.out-01   =>    halui.mdi-command-%02d"% (othercmds +1)
                print >>file, "net rapid-away-cmd       classicladder.0.out-02   =>    halui.mdi-command-%02d"% (othercmds +2)

        gvcp_options_filename = os.path.join(base, "gvcp_options.hal")
        gvcp_call_filename  = os.path.join(base, "gvcp_call_list.hal")
        if self.d.gladevcp:
                # write the call_list
                # the call_list allows multiple hal files to be loaded post gladevcp
                # this simplifies the problem of overwriting the users custom HAL code
                # when they change gvcp sample options
                # if the user asks for existing instead of sample then if the call_list file exists
                # don't overwrite it
            if (not self.d.gladesample and not os.path.exists(gvcp_call_filename)) or self.d.gladesample:
                f1 = open(gvcp_call_filename, "w")
                print >>f1, _("# These files are loaded post gladeVCP, in the order they appear")
                print >>f1
                if self.d.gladesample:
                    print >>f1, "source gvcp_options.hal"
                print >>f1, "source custom_gvcp.hal"
                # write hal file for sample options selected
            if self.d.gladesample:
                    # copy glade panel from temp file to config
                gvcp = os.path.join(base, "gvcp-panel.ui")
                if os.path.exists(gvcp):
                    writebackup(gvcp)
                shutil.copy2('/tmp/gvcp-panel.ui', gvcp)
                    # write gvcp options HAL file
                f1 = open(gvcp_options_filename, "w")
                print >>f1, _("# _DO NOT_ include your HAL commands here.")
                print >>f1, _("# Put custom HAL commands in custom_gvcp.hal")
                print >> f1 
                if self.d.spindlespeedbar:
                    print >>f1, _("# **** Setup of spindle speed display using gladevcp ****")
                    print >>f1
                    if spindle_enc:
                        print >>f1, ("net spindle-fb-rpm-abs-filtered       =>   gladevcp.spindle-speed")
                    else:
                        print >>f1, ("net spindle-vel-cmd-rpm-abs    =>    gladevcp.spindle-speed")
                if self.d.spindleatspeed:
                    print >>f1, ("net spindle-at-speed        =>    gladevcp.spindle-at-speed-led")
                i = 0
                print >>f1, _("# **** Setup GLADE MDI buttons ****")
                print >>f1, ("net machine-is-on          =>    gladevcp.button-box-active")
                for temp in(("zerox","zero-x","x"),("zeroy","zero-y","y"),("zeroz","zero-z","z"),("zeroa","zero-a","a")):
                    if self.d[temp[0]]:
                        print >>f1, ("# **** MDI Command %d - %s-axis is specified in the machine named INI file under [HALUI] heading ****"%(i,temp[1]))
                        print >>f1, ("net MDI-%s            gladevcp.%s          =>  halui.mdi-command-%02d")%(temp[0],temp[1],i)
                        if self.d.require_homing:
                            print >>f1, ("net %s-is-homed      =>    gladevcp.%s-active"% (temp[2],temp[1]))
                        else:
                            print >>f1, ("net machine-is-on          =>    gladevcp.%s-active"% (temp[1]))
                        print >>f1
                        i += 1
                if self.d.autotouchz:
                    print >>f1, _("# **** Z axis touch-off button - requires the touch-off classicladder program ****")
                    print >>f1, ("net auto-touch-z      <=    gladevcp.auto-touch-z")
                    print >>f1, ("net MDI-mode          =>    gladevcp.auto-touch-z-active")
                    print >>f1
        else:
            # gvcp was not selected remove any existing related HAl files
            if os.path.exists(gvcp_options_filename):
                os.remove(gvcp_options_filename)
            if os.path.exists(gvcp_call_filename):
                os.remove(gvcp_call_filename)

        if self.d.pyvcp:
            vcp = os.path.join(base, "pyvcp-panel.xml")
            if not os.path.exists(vcp):
                f1 = open(vcp, "w")
                print >>f1, "<?xml version='1.0' encoding='UTF-8'?>"
                print >>f1, "<!-- "
                print >>f1, _("Include your PyVCP panel here.\n")
                print >>f1, "-->"
                print >>f1, "<pyvcp>"
                print >>f1, "</pyvcp>"

        # the jump list allows multiple hal files to be loaded postgui
        # this simplifies the problem of overwritting the users custom HAL code
        # when they change pyvcp sample options
        # if the user picked existing pyvcp option and the postgui_call_list is present
        # don't overwrite it. otherwise write the file.
        calllist_filename = os.path.join(base, "postgui_call_list.hal")
        if (self.d.pyvcpexist and not os.path.exists(calllist_filename)) or not self.d.pyvcpexist:
            f1 = open(calllist_filename, "w")
            print >>f1, _("# These files are loaded post GUI, in the order they appear")
            print >>f1
            if self.d.pyvcp and self.d.pyvcphaltype == 1 and self.d.pyvcpconnect:
                print >>f1, "source pyvcp_options.hal"
            if self.d.serial_vfd:
                if self.d.gs2_vfd:
                    print >>f1, "source gs2_vfd.hal"
                if self.d.mitsub_vfd:
                    print >>f1, "source mitsub_vfd.hal"
            print >>f1, "source custom_postgui.hal"
            f1.close()


        # If the user asked for pyvcp sample panel add the HAL commands too
        pyfilename = os.path.join(base, "pyvcp_options.hal")
        if self.d.pyvcp and self.d.pyvcphaltype == 1 and self.d.pyvcpconnect: # spindle speed display
            f1 = open(pyfilename, "w")
            print >>f1, _("# _DO NOT_ include your HAL commands here.")
            print >>f1, _("# Put custom HAL commands in custom_postgui.hal")
            print >>f1, _("""# The commands in this file are run after the GUI loads""") 
            print >>f1
            print >>f1, _("# **** Setup of spindle speed display using pyvcp -START ****")
            print >>f1
            if spindle_enc:
                print >>f1, ("net spindle-fb-rpm-abs-filtered       =>   pyvcp.spindle-speed")
            else:
                print >>f1, ("net spindle-vel-cmd-rpm-abs    =>    pyvcp.spindle-speed")
            print >>f1, ("net spindle-at-speed        =>    pyvcp.spindle-at-speed-led")
            print >>f1
            print >>f1, _("# **** Setup of spindle speed display using pyvcp -END ****")
            print >>f1
        else:
            if os.path.exists(pyfilename):
                os.remove(pyfilename)

        # pncconf adds a custom.hal and custom_postgui.hal file if one is not present
        for i in ("custom","custom_postgui","shutdown","custom_gvcp"):
            custom = os.path.join(base, i+".hal")
            if not os.path.exists(custom):
                f1 = open(custom, "w")
                print >>f1, ("# Include your %s HAL commands here")%i
                print >>f1, _("# This file will not be overwritten when you run PNCconf again")

        if self.d.frontend == _PD._TOUCHY:# TOUCHY GUI
                touchyfile = os.path.join(base, "touchy.hal")
            #if not os.path.exists(touchyfile):
                f1 = open(touchyfile, "w")
                print >>f1, _("# These commands are required for Touchy GUI")
                print >>f1, ("net cycle-start          =>   touchy.cycle-start")
                print >>f1, ("net abort                =>   touchy.abort")
                print >>f1, ("net single-step          =>   touchy.single-block")
                print >>f1, ("net selected-jog-incr    <=   touchy.jog.wheel.increment")
                print >>f1, ("net axis-selected-count =>   touchy.wheel-counts")
                print >>f1, ("net jog-x-pos  => touchy.jog.continuous.x.positive")
                print >>f1, ("net jog-x-neg  => touchy.jog.continuous.x.negative")
                print >>f1, ("net jog-y-pos  => touchy.jog.continuous.y.positive")
                print >>f1, ("net jog-y-neg  => touchy.jog.continuous.y.negative")
                print >>f1, ("net jog-z-pos  => touchy.jog.continuous.z.positive")
                print >>f1, ("net jog-z-neg  => touchy.jog.continuous.z.negative")
                print >>f1, ("net quillup  => touchy.quill-up")
                for axletter in enumerate(axis_convert):
                    if axletter in self.d.available_axes:
                        print >>f1, "net axis-select-%s   <=   touchy.jog.wheel.%s"% (axletter, axletter)

        # include Automation Direct GS2 VFD HAL file
        fname = os.path.join(base, "gs2_vfd.hal")
        if self.d.serial_vfd and self.d.gs2_vfd:
            self.write_gs2_vfd(fname)
        else:
            if os.path.exists(fname):
                os.remove(fname)

        # include Mitsubishi FR VFD HAL file
        fname = os.path.join(base, "mitsub_vfd.hal")
        if self.d.serial_vfd and self.d.mitsub_vfd:
            self.write_mitsub_vfd(fname)
        else:
            if os.path.exists(fname):
                os.remove(fname)

        if self.d.classicladder: 
           if not self.d.laddername == "custom.clp":
                filename = os.path.join(_PD.DISTDIR, "configurable_options/ladder/%s" % self.d.laddername)
                original = os.path.expanduser("~/linuxcnc/configs/%s/custom.clp" % self.d.machinename)
                if os.path.exists(filename): # check for the master file to copy from 
                  if os.path.exists(original):
                     #print "custom file already exists"
                     writebackup(original)
                     #shutil.copy( original,os.path.expanduser("~/linuxcnc/configs/%s/backups/custom_backup.clp" % self.d.machinename) ) 
                     print "made backup of existing custom"
                  shutil.copy( filename,original)
                  #print "copied ladder program to usr directory"
                  #print"%s" % filename
                else:
                     print "Master or temp ladder files missing from configurable_options dir"
        if self.d.pyvcp and not self.d.pyvcpexist:                
           panelname = os.path.join(_PD.DISTDIR, "configurable_options/pyvcp/%s" % self.d.pyvcpname)
           originalname = os.path.expanduser("~/linuxcnc/configs/%s/pyvcp-panel.xml" % self.d.machinename)
           if os.path.exists(panelname):     
                  if os.path.exists(originalname):
                     #print "custom PYVCP file already exists"
                     writebackup(originalname)
                     #shutil.copy( originalname,os.path.expanduser("~/linuxcnc/configs/%s/backups/pyvcp-panel_backup.xml" % self.d.machinename) ) 
                     print "made backup of existing custom"
                  shutil.copy( panelname,originalname)
                  #print "copied PYVCP program to usr directory"
                  #print"%s" % panelname
           else:
                  print "Master PYVCP file: %s missing from configurable_options dir"% self.d.pyvcpname
        file.close()
        self.d.add_md5sum(filename)

    def write_gs2_vfd(self,filename):
        p = self.d.gs2_vfd_port
        b = self.d.gs2_vfd_baud
        s = int(self.d.gs2_vfd_slave)
        a = int(self.d.gs2_vfd_accel)
        d = int(self.d.gs2_vfd_deaccel)
        f1 = open(filename, "w")
        print >>f1, _("# Generated by PNCconf at %s") % time.asctime()
        print >>f1
        print >>f1,"#    Automation Direct GS2 VFD for spindle control"
        print >>f1
        print >>f1,"#    The communications switches must be set to RS-232C"
        print >>f1,"#    The motor parameters must be set to match the motor"
        print >>f1,"#    P3.00 (Source of Operation Command) must be set to Operation determined by RS-485 interface, 03 or 04"
        print >>f1,"#    P4.00 (Source of Frequency Command) must be set to Frequency determined by RS232C/RS485 communication interface, 05"
        print >>f1,"#    P9.01 (Transmission Speed) must be set to the specified baud, 9600 = 01"
        print >>f1,"""#    P9.02 (Communication Protocol) must be set to "Modbus RTU mode, 8 data bits, no parity, 2 stop bits", 03"""
        print >>f1
        print >>f1, "loadusr -Wn spindle-vfd gs2_vfd -d %s -r %s -p none -s 2 -t %s -n spindle-vfd -A %s -D %s"%(p,b,s,a,d)
        print >>f1, "net spindle-cw           => spindle-vfd.spindle-fwd"
        print >>f1, "net spindle-enable       => spindle-vfd.spindle-on"
        print >>f1, "net spindle-at-speed     <= spindle-vfd.at-speed"
        print >>f1, "net spindle-vel-cmd-rpm  => spindle-vfd.speed-command"
        f1.close()

    def write_mitsub_vfd(self,filename):
        f1 = open(filename, "w")
        print >>f1, _("# Generated by PNCconf at %s") % time.asctime()
        print >>f1
        print >>f1,"#    Misubishi FR VFD for spindle control"
        f1.close()

    def write_readme(self, base):
        filename = os.path.join(base, "README")
        file = open(filename, "w")
        print >>file, _("Generated by PNCconf at %s") % time.asctime()
        print >>file
        if  self.d.units == _PD._IMPERIAL: unit = "an imperial"
        else: unit = "a metric"
        if self.d.frontend == _PD._AXIS: display = "AXIS"
        elif self.d.frontend == _PD._TKLINUXCNC: display = "Tklinuxcnc"
        elif self.d.frontend == _PD._GMOCCAPY: display = "TOUCHY"
        elif self.d.frontend == _PD._TOUCHY: display = "TOUCHY"
        else: display = "an unknown"
        if self.d.axes == 0:machinetype ="XYZ"
        elif self.d.axes == 1:machinetype ="XYZA"
        elif self.d.axes == 2:machinetype ="XZ-Lathe"
        print >>file, self.d.machinename,_("configures LinuxCNC as:\n")
        print >>file, unit,machinetype,_("type CNC\n")
        print >>file, display,_("will be used as the frontend display")
        print >>file
        if self.d.number_mesa <> 0:
            for boardnum in range(0,int(self.d.number_mesa)):
                print >>file, "Mesa hardware I/O card - board %d is designated as\n"% boardnum,self.d["mesa%d_currentfirmwaredata"% boardnum][_PD._BOARDTITLE] 
                print >>file, "with", self.d["mesa%d_currentfirmwaredata"% boardnum][_PD._MAXGPIO], "I/O pins and firmware is:", self.d["mesa%d_firmware"% boardnum]
                print >>file
            for boardnum in range(0,int(self.d.number_mesa)):
                for concount,connector in enumerate(self.d["mesa%d_currentfirmwaredata"% boardnum][_PD._NUMOFCNCTRS]) :
                    print >>file,"** Mesa %s -> Board #"% self.d["mesa%d_boardtitle"% boardnum],boardnum,_(" connector")," %d **\n"% connector
                    print >>file ,"connector    board     num   component    signal        "
                    print >>file ,"number    position           type        name\n"
                    conpin = 1
                    for pin in range (0,24):
                        dummy,compnum = self.d["mesa%d_currentfirmwaredata"% (boardnum)][_PD._STARTOFDATA+pin+(concount*24)]
                        temp = self.d["mesa%dc%dpin%d" % (boardnum,connector,pin) ]
                        tempinv = self.d["mesa%dc%dpin%dinv" % (boardnum,connector,pin) ]
                        temptype = self.d["mesa%dc%dpin%dtype" % (boardnum,connector,pin) ]
                        if temptype in(_PD.GPIOI,_PD.GPIOO,_PD.GPIOD):
                            compnum = pin+(concount*24)
                        if tempinv: 
                            invmessage = _("invrt")
                        else: invmessage =""
                        print >>file, ("P%d-%d   %d   (%d %s)  %s  %s\n"%(connector,conpin,pin,compnum,temptype,temp,invmessage))
                        #print >>file, ("    connected to signal:'%(data)s'%(mess)s\n" %{'data':temp, 'mess':invmessage})
                        conpin +=2
            print >>file
        templist = ("pp1","pp2","pp3")
        for j, k in enumerate(templist):
            if self.d.number_pports < (j+1): break 
            print >>file, _("%(name)s Parport" % { 'name':k})
            for x in (2,3,4,5,6,7,8,9,10,11,12,13,15): 
                temp = self.d["%sIpin%d" % (k, x)]
                tempinv = self.d["%sIpin%dinv" % (k, x)]
                if tempinv: 
                    invmessage = _("-> inverted")
                else: invmessage =""
                print >>file,_("pin# %(pinnum)d is connected to input signal:'%(data)s' %(mesag)s" 
                %{ 'pinnum':x,'data':temp,'mesag':invmessage})          
            for x in (1,2,3,4,5,6,7,8,9,14,16,17):  
                temp = self.d["%sOpin%d" % (k, x)]
                tempinv = self.d["%sOpin%dinv" % (k, x)]
                if tempinv: 
                    invmessage = _("-> inverted")
                else: invmessage =""
                print >>file,_("pin# %(pinnum)d is connected to output signal:'%(data)s' %(mesag)s" 
                %{ 'pinnum':x,'data':temp,'mesag':invmessage})   
            print >>file 
        file.close()
        self.d.add_md5sum(filename)

#*******************
# helper functions
#*******************

    def build_pid(self, file, jnum, let, stepflag):
        title = 'JOINT'
        if let == 's':
            title = 'SPINDLE'
        print >>file, "setp   pid.%s.Pgain     [%s_%d]P" % (let, title, jnum)
        print >>file, "setp   pid.%s.Igain     [%s_%d]I" % (let, title, jnum)
        print >>file, "setp   pid.%s.Dgain     [%s_%d]D" % (let, title, jnum)
        print >>file, "setp   pid.%s.bias      [%s_%d]BIAS" % (let, title, jnum)
        print >>file, "setp   pid.%s.FF0       [%s_%d]FF0" % (let, title, jnum)
        print >>file, "setp   pid.%s.FF1       [%s_%d]FF1" % (let, title, jnum)
        print >>file, "setp   pid.%s.FF2       [%s_%d]FF2" % (let, title, jnum)
        print >>file, "setp   pid.%s.deadband  [%s_%d]DEADBAND" % (let, title, jnum)
        if let =='s' and self.d.suseoutputrange2:
            print >>file, "net ratio_select.out   pid.%s.maxoutput " % (let)
        else:
            print >>file, "setp   pid.%s.maxoutput [%s_%d]MAX_OUTPUT" % (let, title, jnum)
        # steppers
        print >>file, "setp   pid.%s.error-previous-target true" % let
        # steppers
        if stepflag:
            if  self.d.units == _PD._IMPERIAL:
                me = .0005
            else:
                me = .0127
            print >>file, ('# This setting is to limit bogus stepgen')
            print >>file, ('# velocity corrections caused by position')
            print >>file, ('# feedback sample time jitter.')
            print >>file, "setp   pid.%s.maxerror %f" % (let, me)
        print >>file
        if let == 's':
            name = "spindle"
        else:
            name = let
        print >>file, "net %s-index-enable  <=> pid.%s.index-enable" % (name, let)
        print >>file, "net %s-enable        =>  pid.%s.enable" % (name, let)

        if let == 's':
            if self.d.susenegativevoltage:
                signal = "spindle-vel-cmd-rpm"
                fbsignal= "spindle-vel-fb-rpm"
            else:
                signal = "spindle-vel-cmd-rpm-abs"
                fbsignal= "spindle-vel-fb-rpm-abs"
            print >>file, "net %s     => pid.%s.command" % (signal, let)
            print >>file, "net %s      => pid.%s.feedback"% (fbsignal, let)
            if self.d.suseoutputrange2:
                print >>file, "net spindle-pid-out  pid.s.output    => scale.gear.in"
                print >>file, "net gear-ratio       ratio_select.out-f => scale.gear.gain"
                print >>file, "setp ratio_select.in00 %f" % (1/float(self.d.gsincrvalue0))
                print >>file, "setp ratio_select.in01 %f" % (1/float(self.d.gsincrvalue1))
                #print >>file, "setp ratio_select.in2 %f" % self.d.gsincrvalue2
                #print >>file, "setp ratio_select.in4 %f" % self.d.gsincrvalue4
                print >>file, "net gear-select-a         =>  ratio_select.sel0"
                #print >>file, "net gear-select-b ratio_select.sel1"
                #print >>file, "net gear-select-c ratio_select.sel2"
                print >>file, "net spindle-output        <=  scale.gear.out"
            else:
                print >>file, "net spindle-output        <=  pid.%s.output"% (let)
        else:
           print >>file, "net %s-pos-cmd       =>  pid.%s.command" % (name, let)
           #print >>file, "net %s-vel-cmd       =>  pid.%s.command-deriv" % (name, let) # This must be connected to something
           print >>file, "net %s-pos-fb        =>  pid.%s.feedback"% (name,let)
           print >>file, "net %s-output        <=  pid.%s.output"% (name, let)
           #print >>file, "net %s-vel-fb           => pid.%s.feedback-deriv"% (name, let) # This must be connected to something
        print >>file

    def connect_joint(self, file, jnum, let):
        def get(s): return self.d[let + s]
        title = 'JOINT'
        if let == 's':
            title = 'SPINDLE'
        closedloop = False
        pwmpin = self.a.pwmgen_sig(let)
        pwmpinname = self.a.make_pinname(pwmpin, substitution = self.d.useinisubstitution)
        if pwmpinname and not 'serial' in pwmpin: # TODO allow sserial PWM to be inverted
            pwminvertlist = self.a.pwmgen_invert_pins(pwmpin)
        if not pwmpin == None:
            pwmtype = self.a.pwmgen_sig(let)+"type"
        else:
            pwmtype = None
        tppwmpinname = self.a.make_pinname(self.a.tppwmgen_sig(let), substitution = self.d.useinisubstitution)
        tppwm_six = self.a.tppwmgen_has_6(let)
        steppinname = self.a.make_pinname(self.a.stepgen_sig(let), substitution = self.d.useinisubstitution)
        try:
            bldc_control = self.d[let+"bldc_option"]
        except:
            bldc_control = False
        if steppinname:
            stepinvertlist = self.a.stepgen_invert_pins(self.a.stepgen_sig(let))
        encoderpinname = self.a.make_pinname(self.a.encoder_sig(let), substitution = self.d.useinisubstitution)
        amp8i20pinname = self.a.make_pinname(self.a.amp_8i20_sig(let), substitution = self.d.useinisubstitution)
        resolverpinname = self.a.make_pinname(self.a.resolver_sig(let), substitution = self.d.useinisubstitution)
        potpinname = self.a.make_pinname(self.a.potoutput_sig(let), substitution = self.d.useinisubstitution)
        if potpinname:
            potinvertlist = self.a.spindle_invert_pins(self.a.potoutput_sig(let))
        if steppinname and encoderpinname and not let == 's': closedloop = True
        if (encoderpinname or resolverpinname) and (pwmpinname or tppwmpinname or amp8i20pinname): closedloop = True
        print let + " is closedloop? "+ str(closedloop)
        print " ENCODER:",encoderpinname," RESOLVER:",resolverpinname
        print " PWM:",pwmpinname," 3PWM:",tppwmpinname," 8i20:",amp8i20pinname
        print " STEPPER:",steppinname
        print " POTENTIOMETER:",potpinname
        lat = self.d.latency
        print >>file, "#*******************"
        if let.upper() == 'S':
            print >>file, "#  SPINDLE"
        elif len(let) >1:
            print >>file, "#  Tandem AXIS %s %s %d" % (let.upper(), title, jnum )
        else:
            print >>file, "#  AXIS %s %s %d" % (let.upper(), title, jnum )
        print >>file, "#*******************"
        print >>file

        if bldc_control:
            bldc = self.d[let+"bldc_config"]
            print >>file, "# -- BLDC setup --"
            print >>file, "setp   bldc.%d.drive-offset       %d" % (jnum,self.d[let+"bldc_drive_offset"])
            print >>file, "setp   bldc.%s.rev                %d" % (jnum,self.d[let+"bldc_reverse"])
            if "q" in(bldc):
                print >>file, "setp   bldc.%d.scale              %d" % (jnum,self.d[let+"bldc_scale"])
                print >>file, "setp   bldc.%d.poles              %d" % (jnum,self.d[let+"bldc_poles"])
            if "i" in(bldc):
                print >>file, "setp   bldc.%s.initvalue          %d" % (jnum,self.d[let+"bldc_inital_value"])
            if "i" in(bldc) or "a" in(bldc):
                print >>file, "setp   bldc.%s.lead-angle         %d" % (jnum,self.d[let+"bldc_lead_angle"])
                print >>file, "setp   bldc.%d.encoder-offset     %d" % (jnum,self.d[let+"bldc_encoder_offset"])
            if "h" in(bldc):
                print >>file, "setp   bldc.%d.pattern            %d" % (jnum,self.d[let+"bldc_pattern_in"])
                print >>file, "net %s-hall1-in      bldc.%d.hall1"% (let,jnum)
                print >>file, "net %s-hall2-in      bldc.%d.hall2"% (let,jnum)
                print >>file, "net %s-hall3-in      bldc.%d.hall3"% (let,jnum)
            if "f" in(bldc):
                print >>file, "net %s-c1-in     bldc.%d.C1"% (let,jnum)
                print >>file, "net %s-c2-in     bldc.%d.C2"% (let,jnum)
                print >>file, "net %s-c4-in     bldc.%d.C4"% (let,jnum)
                print >>file, "net %s-c8-in     bldc.%d.C8"% (let,jnum)
            if "H" in(bldc):
                print >>file, "setp   bldc.%d.output-pattern     %d" % (jnum,self.d[let+"bldc_pattern_out"])
                print >>file, "net %s-hall1-out     bldc.%d.hall1-out"% (let,jnum)
                print >>file, "net %s-hall2-out     bldc.%d.hall2-out"% (let,jnum)
                print >>file, "net %s-hall3-out     bldc.%d.hall3-out"% (let,jnum)
            if "6" in(bldc) :
                if "B" in(bldc):
                    print >>file, "net %s-a-high-on     bldc.%d.A-high-on"% (let,jnum)
                    print >>file, "net %s-a-low-on      bldc.%d.A-low-on"% (let,jnum)
                    print >>file, "net %s-b-high-on     bldc.%d.B-high-on"% (let,jnum)
                    print >>file, "net %s-b-low-on      bldc.%d.B-low-on"% (let,jnum)
                    print >>file, "net %s-c-high-on     bldc.%d.C-high-on"% (let,jnum)
                    print >>file, "net %s-c-low-on      bldc.%d.C-low-on"% (let,jnum)
                else:
                    print >>file, "net %s-a-high-value      bldc.%d.A-high"% (let,jnum)
                    print >>file, "net %s-a-low-value       bldc.%d.A-low"% (let,jnum)
                    print >>file, "net %s-b-high-value      bldc.%d.B-high"% (let,jnum)
                    print >>file, "net %s-b-low-value       bldc.%d.B-low"% (let,jnum)
                    print >>file, "net %s-c-high-value      bldc.%d.C-high"% (let,jnum)
                    print >>file, "net %s-c-low-value       bldc.%d.C-low"% (let,jnum)
            elif "B" in(bldc):
                print >>file, "net %s-a-on          bldc.%d.A-on"% (let,jnum)
                print >>file, "net %s-b-on          bldc.%d.B-on"% (let,jnum)
                print >>file, "net %s-c-on          bldc.%d.C-on"% (let,jnum)
            elif "F" in(bldc):
                print >>file, "net %s-c1-out        bldc.%d.C1-out"% (let,jnum)
                print >>file, "net %s-c2-out        bldc.%d.C2-out"% (let,jnum)
                print >>file, "net %s-c4-out        bldc.%d.C4-out"% (let,jnum)
                print >>file, "net %s-c8-out        bldc.%d.C8-out"% (let,jnum)
            else:
                print >>file, "net %s-a-value       bldc.%d.A-value"% (let,jnum)
                print >>file, "net %s-b-value       bldc.%d.B-value"% (let,jnum)
                print >>file, "net %s-c-value       bldc.%d.C-value"% (let,jnum)
            print >>file
            print >>file, "net %s-pos-rawcounts      bldc.%d.rawcounts"% (let,jnum)
            print >>file, "net %s-index-enable      bldc.%d.index-enable"% (let,jnum)
            print >>file, "net %s-bldc-current      bldc.%d.out"% (let,jnum)
            print >>file, "net %s-meas-angle        bldc.%d.phase-angle"% (let,jnum)
            print >>file, "net %s-output             bldc.%d.value"% (let,jnum)
            print >>file, "net %s-enable             bldc.%d.init"% (let,jnum)
            print >>file, "net %s-is-init           bldc.%s.init-done"% (let,jnum)
            print >>file

        stepflag = bool(steppinname !=None)
        self.build_pid(file, jnum, let, stepflag)

        if tppwmpinname:
                print >>file, "# ---TPPWM Generator signals/setup---"
                if tppwm_six:
                    print >>file, "# six output 3pwg"
                else:print >>file, "# three output 3pwg"
                print >>file, "# TODO write some commands!"
                print >>file

        if amp8i20pinname:
                print >>file, "# ---8i20 amplifier card signals/setup---"
                print >>file
                print >>file, "setp       %s.max_current %.3f"% (amp8i20pinname,self.d[let+"8i20maxcurrent"])
                print >>file, "net %s-meas-angle =>       %s.angle"% (let,amp8i20pinname)
                print >>file, "net %s-bldc-current =>     %s.current"% (let,amp8i20pinname)
                print >>file, "net %s-enable =>            %s.amp_enable"% (let,amp8i20pinname)
                print >>file

        if potpinname:
                # sserial digital potentiometer outputs for spindle eg 7i76 board
                print >>file, "# ---digital potentionmeter output signals/setup---"
                print >>file
                print >>file, "setp   "+potpinname+"spinout-minlim    [%s_%d]OUTPUT_MIN_LIMIT"% (title, jnum)
                print >>file, "setp   "+potpinname+"spinout-maxlim    [%s_%d]OUTPUT_MAX_LIMIT"% (title, jnum)
                print >>file, "setp   "+potpinname+"spinout-scalemax  [%s_%d]OUTPUT_SCALE"% (title, jnum)
                for i in potinvertlist:
                    if i == _PD.POTO:
                        print >>file, "setp   "+potpinname+"spindir-invert   true"
                    if i == _PD.POTE:
                        print >>file, "setp   "+potpinname+"spinena-invert   true"
                print >>file
                print >>file, "net spindle-output      => " + potpinname + "spinout"
                print >>file, "net spindle-enable      => " + potpinname +"spinena"
                print >>file, "net spindle-ccw         => " + potpinname +"spindir"
                print >>file

        if pwmpinname:
            print >>file, "# ---PWM Generator signals/setup---"
            print >>file
            # sserial daughter board PWMGENS eg 7i77
            if "analogout" in pwmpinname:
                rawpinname = self.a.make_pinname(pwmpin, gpionumber = False, prefixonly = True, substitution = self.d.useinisubstitution) # dont want the component name

                if let == 's':
                    print >>file, "setp   "+pwmpinname+"-scalemax  [%s_%d]OUTPUT_SCALE"% (title, jnum)
                    print >>file, "setp   "+pwmpinname+"-minlim    [%s_%d]OUTPUT_MIN_LIMIT"% (title, jnum)
                    print >>file, "setp   "+pwmpinname+"-maxlim    [%s_%d]OUTPUT_MAX_LIMIT"% (title, jnum)
                    print >>file
                    print >>file, "net spindle-output      => " + pwmpinname
                    if 'analogout5' in pwmpinname: # on the 7i77 analog out 5 has it's own enable
                        print >>file, "net spindle-enable      => " + rawpinname + "spinena"
                else:
                    print >>file, "setp   "+pwmpinname+"-scalemax  [%s_%d]OUTPUT_SCALE"% (title, jnum)
                    print >>file, "setp   "+pwmpinname+"-minlim    [%s_%d]OUTPUT_MIN_LIMIT"% (title, jnum)
                    print >>file, "setp   "+pwmpinname+"-maxlim    [%s_%d]OUTPUT_MAX_LIMIT"% (title, jnum)
                    print >>file
                    print >>file, "net %s-output     => "% (let) + pwmpinname
                    print >>file, "net %s-pos-cmd    <= joint.%d.motor-pos-cmd" % (let, jnum )
                    print >>file, "net %s-enable     <= joint.%d.amp-enable-out"% (let,jnum)
                    if 'analogout5' in pwmpinname: # on the 7i77 analog out 5 has it's own enable
                        print >>file, "net %s-enable   %spinena"% (let,rawpinname)
                    if let == "x":
                        print >>file, "# enable _all_ sserial pwmgens"
                        print >>file, "net %s-enable   => %sanalogena"% (let,rawpinname)
                print >>file

            else:
                # mainboard PWMGENS
                pulsetype = 1
                if self.d[pwmtype] == _PD.PDMP: pulsetype = 3
                if self.d[pwmtype] == _PD.UDMU: pulsetype = 2
                print >>file, "setp   "+pwmpinname+".output-type %d"% pulsetype 
                print >>file, "setp   "+pwmpinname+".scale  [%s_%d]OUTPUT_SCALE"% (title, jnum)
                for i in pwminvertlist:
                    print >>file, "setp    "+i+".invert_output true"
                print >>file
                if let == 's':  
                    print >>file
                    if closedloop or self.d.suseoutputrange2:
                        print >>file, "net spindle-output      => " + pwmpinname + ".value"
                        print >>file, "net spindle-enable      => " + pwmpinname +".enable"    
                    else:
                        print >>file, "net spindle-vel-cmd-rpm     => " + pwmpinname + ".value"
                        print >>file, "net spindle-enable      => " + pwmpinname +".enable"
                else:
                    print >>file, "net %s-output                             => "% (let) + pwmpinname + ".value"
                    print >>file, "net %s-pos-cmd    joint.%d.motor-pos-cmd" % (let, jnum )
                    print >>file, "net %s-enable     joint.%d.amp-enable-out  => "% (let,jnum) + pwmpinname +".enable"
                print >>file

        if steppinname:
            if len(let) >1:
                print >>file, "# Step Gen signals/setup for tandem axis"
            else:
                print >>file, "# Step Gen signals/setup"
            print >>file
            print >>file, "setp   " + steppinname + ".dirsetup        [%s_%d]DIRSETUP"% (title, jnum)
            print >>file, "setp   " + steppinname + ".dirhold         [%s_%d]DIRHOLD"% (title, jnum)
            print >>file, "setp   " + steppinname + ".steplen         [%s_%d]STEPLEN"% (title, jnum)
            print >>file, "setp   " + steppinname + ".stepspace       [%s_%d]STEPSPACE"% (title, jnum)
            print >>file, "setp   " + steppinname + ".position-scale  [%s_%d]STEP_SCALE"% (title, jnum)
            print >>file, "setp   " + steppinname + ".step_type        0"
            print >>file, "setp   " + steppinname + ".control-type     1"
            if let =="s":
                print >>file, "setp   " + steppinname + ".maxaccel         [%s_%d]MAX_ACCELERATION"% (title, jnum)
                print >>file, "setp   " + steppinname + ".maxvel           [%s_%d]MAX_VELOCITY"% (title, jnum)
            else:
                print >>file, "setp   " + steppinname + ".maxaccel         [%s_%d]STEPGEN_MAXACCEL"% (title, jnum)
                print >>file, "setp   " + steppinname + ".maxvel           [%s_%d]STEPGEN_MAXVEL"% (title, jnum)

            # invert step pins if requested
            # step does not have alias pin names so we invert it's GPIO pin
            for i in stepinvertlist[0]:
                   print >>file, "setp   " + steppinname + ".step.invert_output   true"
            # step direction has an alias pin
            for i in stepinvertlist[1]:
                   print >>file, "setp   " + steppinname + ".direction.invert_output   true"
            if let == "s":
                print >>file
                print >>file, "net spindle-enable          =>  " + steppinname + ".enable" 
                print >>file, "net spindle-vel-cmd-rps     =>  "+ steppinname + ".velocity-cmd"
                if not encoderpinname and not resolverpinname:
                    print >>file, "net spindle-vel-fb-rps         <=  "+ steppinname + ".velocity-fb"
            else:
                print >>file
                print >>file, "# ---closedloop stepper signals---"
                print >>file
                print >>file, "net %s-pos-cmd    <= joint.%d.motor-pos-cmd" % (let, jnum )
                print >>file, "net %s-vel-cmd    <= joint.%d.vel-cmd" % (let, jnum )
                print >>file, "net %s-output     <= "% (let) + steppinname + ".velocity-cmd"
                print >>file, "net %s-pos-fb     <= "% (let) + steppinname + ".position-fb"
                print >>file, "net %s-pos-fb     => joint.%d.motor-pos-fb" % (let, jnum )
                print >>file, "net %s-enable     <= joint.%d.amp-enable-out"% (let,jnum)
                print >>file, "net %s-enable     => %s.enable"% (let, steppinname)
                print >>file

        if encoderpinname:             
            countmode = 0
            if let == "s" and self.d.ssingleinputencoder: countmode = 1
            print >>file, "# ---Encoder feedback signals/setup---"
            print >>file
            print >>file, "setp    "+encoderpinname+".counter-mode %d"% countmode
            print >>file, "setp    "+encoderpinname+".filter 1"
            print >>file, "setp    "+encoderpinname+".index-invert 0"
            print >>file, "setp    "+encoderpinname+".index-mask 0"
            print >>file, "setp    "+encoderpinname+".index-mask-invert 0"
            print >>file, "setp    "+encoderpinname+".scale  [%s_%d]ENCODER_SCALE"% (title, jnum)
            print >>file
            if let == 's':
                print >>file, "net spindle-revs             <=   " + encoderpinname + ".position"
                print >>file, "net spindle-vel-fb-rps       <=   " + encoderpinname + ".velocity"
                print >>file, "net spindle-index-enable     <=>  " + encoderpinname + ".index-enable"
            else:
                print >>file, "net %s-pos-fb               <=  "% (let) + encoderpinname+".position"
                print >>file, "net %s-vel-fb               <=  "% (let) + encoderpinname + ".velocity"
                print >>file, "net %s-pos-fb               =>  joint.%d.motor-pos-fb" % (let, jnum)
                print >>file, "net %s-index-enable    joint.%d.index-enable  <=>  "% (let, jnum) + encoderpinname + ".index-enable"
                print >>file, "net %s-pos-rawcounts        <=  "% (let) + encoderpinname + ".rawcounts"
            print >>file

        if resolverpinname:
            print >>file, "# ---Resolver feedback signals/setup---"
            print >>file
            print >>file, "setp    "+resolverpinname+".velocity-scale 1 # mptor speed in RPS"
            print >>file, "setp    "+resolverpinname+".scale  [%s_%d]RESOLVER_SCALE"% (title, jnum)
            print >>file
            print >>file, "net %s-pos-rawcounts        <=  "% (let) + resolverpinname + ".rawcounts"
            if let == 's':
                print >>file, "net spindle-revs             <=   " + resolverpinname + ".position"
                print >>file, "net spindle-vel-fb-rps       <=   " + resolverpinname + ".velocity"
                print >>file, "net spindle-index-enable     <=>  " + resolverpinname + ".index-enable"
            else:
                print >>file, "net %s-pos-fb               <=  "% (let) + resolverpinname+".position"
                print >>file, "net %s-vel-fb               <=  "% (let) + resolverpinname + ".velocity"
                print >>file, "net %s-pos-fb               =>  joint.%d.motor-pos-fb" % (let, jnum)
                print >>file, "net %s-index-enable    joint.%d.index-enable  <=>  "% (let, jnum) + resolverpinname + ".index-enable"
            print >>file

        if let =='s':
            print >>file, "# ---setup spindle control signals---" 
            print >>file
            print >>file, "net spindle-vel-cmd-rps        <=  spindle.0.speed-out-rps"
            print >>file, "net spindle-vel-cmd-rps-abs    <=  spindle.0.speed-out-rps-abs"
            print >>file, "net spindle-vel-cmd-rpm        <=  spindle.0.speed-out"
            print >>file, "net spindle-vel-cmd-rpm-abs    <=  spindle.0.speed-out-abs"
            print >>file, "net spindle-enable             <=  spindle.0.on"
            print >>file, "net spindle-cw                 <=  spindle.0.forward"
            print >>file, "net spindle-ccw                <=  spindle.0.reverse"
            print >>file, "net spindle-brake              <=  spindle.0.brake"
            print >>file, "net spindle-revs               =>  spindle.0.revs"
            print >>file, "net spindle-at-speed           =>  spindle.0.at-speed"
            print >>file, "net spindle-vel-fb-rps         =>  spindle.0.speed-in"
            print >>file, "net spindle-index-enable      <=>  spindle.0.index-enable"
            print >>file
            if not self.a.findsignal("spindle-at-speed"):
                print >>file, "# ---Setup spindle at speed signals---"
                print >>file
                if (encoderpinname or resolverpinname) and self.d.suseatspeed:
                    if self.d.susenearrange:
                        near_scale = 1
                        near_range = self.d.snearrange/60
                    else:
                        near_scale =  self.d.snearscale
                        near_range = 0
                    print >>file, "net spindle-vel-cmd-rps    =>  near.0.in1"
                    print >>file, "net spindle-vel-fb-rps         =>  near.0.in2"
                    print >>file, "net spindle-at-speed       <=  near.0.out"
                    print >>file, "setp near.0.scale %f"%near_scale
                    print >>file, "setp near.0.difference %f"% near_range
                    print >>file
                else:
                    print >>file, "sets spindle-at-speed true"
                    print >>file
            if encoderpinname or resolverpinname:
                if (self.d.pyvcp and self.d.pyvcpconnect == 1 and self.d.pyvcphaltype == 1) \
                  or self.d.gladevcp and self.d.spindlespeedbar:
                    print >>file, _("#  Use ACTUAL spindle velocity from spindle encoder")
                    print >>file, _("#  spindle-velocity bounces around so we filter it with lowpass")
                    print >>file, _("#  spindle-velocity is signed so we use absolute component to remove sign") 
                    print >>file, _("#  ACTUAL velocity is in RPS not RPM so we scale it.")
                    print >>file
                    print >>file, ("setp     scale.spindle.gain 60")
                    print >>file, ("setp     lowpass.spindle.gain %f"% self.d.sfiltergain)
                    print >>file, ("net spindle-vel-fb-rps        =>     scale.spindle.in")
                    print >>file, ("net spindle-fb-rpm               scale.spindle.out       =>   abs.spindle.in")
                    print >>file, ("net spindle-fb-rpm-abs           abs.spindle.out         =>   lowpass.spindle.in")
                    print >>file, ("net spindle-fb-rpm-abs-filtered  lowpass.spindle.out  ")
            return

        min_limsig = self.a.min_lim_sig(let)
        if not min_limsig: min_limsig = "%s-neg-limit" % let
        max_limsig = self.a.max_lim_sig(let)  
        if not max_limsig: max_limsig = "%s-pos-limit" % let 
        homesig = self.a.home_sig(let)
        if not homesig: homesig = "%s-home-sw" % let
        print >>file, "# ---setup home / limit switch signals---"       
        print >>file       
        print >>file, "net %s     =>  joint.%d.home-sw-in" % (homesig, jnum)       
        print >>file, "net %s     =>  joint.%d.neg-lim-sw-in" % (min_limsig, jnum)       
        print >>file, "net %s     =>  joint.%d.pos-lim-sw-in" % (max_limsig, jnum)
        print >>file                

    def connect_input(self, file):
        print >>file, "# external input signals"

        def write_pins(pname,p,i,t):
            # for input pins
            if t == _PD.GPIOI:
                if not p == "unused-input":
                    pinname = self.a.make_pinname(pname, substitution = self.d.useinisubstitution) 
                    print >>file, "\n# ---",p.upper(),"---"
                    if "parport" in pinname:
                        if i: print >>file, "net %s     <= %s-not" % (p, pinname)
                        else: print >>file, "net %s     <= %s" % (p, pinname)
                    elif "sserial" in pname:
                        if i: print >>file, "net %s     <=  "% (p)+pinname +"-not"
                        else: print >>file, "net %s     <=  "% (p)+pinname
                    else:
                        if i: print >>file, "net %s     <=  "% (p)+pinname +".in_not"
                        else: print >>file, "net %s     <=  "% (p)+pinname +".in"
            # for encoder pins
            elif t in (_PD.ENCA,_PD.MXE0,_PD.MXE1):
                if not p == "unused-encoder":
                    for sig in (self.d.halencoderinputsignames):
                       if p == sig+"-a":
                            pinname = self.a.make_pinname(self.a.findsignal( p ), substitution = self.d.useinisubstitution)
                            print >>file, "\n# ---",sig.upper(),"---"
                            print >>file, "net %s         <=  "% (sig+"-position")+pinname +".position"
                            print >>file, "net %s            <=  "% (sig+"-count")+pinname +".count"
                            print >>file, "net %s         <=  "% (sig+"-velocity")+pinname +".velocity"
                            print >>file, "net %s            <=  "% (sig+"-reset")+pinname +".reset"
                            print >>file, "net %s     <=  "% (sig+"-index-enable")+pinname +".index-enable"
                            for ending in ("counter-mode","filter","index-invert","indek-mask","index-mask-invert","scale","vel-timeout"):
                                title = sig + "-%s"% ending
                                name = pinname + ".%s"% ending
                                self._substitution_list.append((title.upper(),name))
                            self._substitution_list.append(("",""))
                            break
            elif t in (_PD.RES0,_PD.RES1,_PD.RES2,_PD.RES3,_PD.RES4,_PD.RES5):
                if not p == "unused-resolver":
                    for sig in (self.d.halresolversignames):
                       if p == sig:
                            pinname = self.a.make_pinname(self.a.findsignal( p ), substitution = self.d.useinisubstitution)
                            print >>file, "\n# ---",sig.upper(),"---"
                            print >>file, "net %s         <=  "% (sig+"-position")+pinname +".position"
                            print >>file, "net %s            <=  "% (sig+"-count")+pinname +".count"
                            print >>file, "net %s            <=  "% (sig+"-angle")+pinname +".angle"
                            print >>file, "net %s            <=  "% (sig+"-error")+pinname +".error"
                            print >>file, "net %s         <=  "% (sig+"-velocity")+pinname +".velocity"
                            print >>file, "net %s            <=  "% (sig+"-reset")+pinname +".reset"
                            print >>file, "net %s     <=  "% (sig+"-index-enable")+pinname +".index-enable"
                            for ending in ("scale","velocity-scale"):
                                title = sig + "-%s"% ending
                                name = pinname + ".%s"% ending
                                self._substitution_list.append((title.upper(),name))
                            self._substitution_list.append(("",""))
                            break
            # for analog in pins
            elif t == (_PD.ANALOGIN):
                if not p == "unused-analog-input":
                            pinname = self.a.make_pinname(self.a.findsignal( p ), substitution = self.d.useinisubstitution)
                            print >>file, "\n# ---",p.upper(),"---"
                            print >>file, "net %s         <=  "% (p)+pinname

        # mesa mainboards
        for boardnum in range(0,int(self.d.number_mesa)):
            for concount,connector in enumerate(self.d["mesa%d_currentfirmwaredata"% (boardnum)][_PD._NUMOFCNCTRS]) :
                for pin in range(0,24):
                    pname = 'mesa%dc%dpin%d' % (boardnum,connector, pin)
                    p = self.d['mesa%dc%dpin%d' % (boardnum,connector, pin)]
                    i = self.d['mesa%dc%dpin%dinv' % (boardnum,connector, pin)]
                    t = self.d['mesa%dc%dpin%dtype' % (boardnum,connector, pin)]
                    write_pins(pname,p,i,t)
        # sserial
            if self.d["mesa%d_numof_sserialports"% (boardnum)]: # only check if we have sserialports
                print >>file
                port = 0
                for channel in range (0,self.d["mesa%d_currentfirmwaredata"% boardnum][_PD._MAXSSERIALCHANNELS]):
                    if channel >4: break # TODO only have 5 channels worth of glade widgets
                    for pin in range (0,_PD._SSCOMBOLEN):
                        pname = 'mesa%dsserial%d_%dpin%d' % (boardnum,port,channel,pin)
                        p = self.d['mesa%dsserial%d_%dpin%d' % (boardnum,port,channel,pin)]
                        i = self.d['mesa%dsserial%d_%dpin%dinv' % (boardnum,port,channel,pin)]
                        t = self.d['mesa%dsserial%d_%dpin%dtype' % (boardnum,port,channel,pin)]
                        write_pins(pname,p,i,t)
        # parports
        templist = ("pp1","pp2","pp3")
        for j, k in enumerate(templist):
            if self.d.number_pports < (j+1): break
            print >>file
            for x in (2,3,4,5,6,7,8,9,10,11,12,13,15):
                pname = "%s_Ipin%d" % (k, x)
                p = self.d[pname]
                i = self.d[pname+"_inv"]
                if not p == "unused-input":
                    write_pins(pname,p,i,_PD.GPIOI)

    def connect_output(self, file):
        print >>file, "# external output signals"

        def write_pins(pname,p,i,t,boardnum,connector,port,channel,pin):
            # for output /open drain pins
            if t in (_PD.GPIOO,_PD.GPIOD,_PD.SSR0):
                if not p == "unused-output":
                    pinname = self.a.make_pinname(pname, substitution = self.d.useinisubstitution)
                    print >>file, "\n# ---",p.upper(),"---"
                    if "parport" in pinname:
                        if p == "force-pin-true":
                            print >>file, "setp %s true"% (pinname)
                        else:
                            print >>file, "net %s  =>     %s"% (p,pinname)
                    else:
                        if "sserial" in pname:
                            temp = pinname
                        # mainboard GPIOO require extra setup commands
                        else:
                            if not t == _PD.SSR0: print >>file, "setp %s true"% (pinname + ".is_output")
                            if t == _PD.GPIOD: print >>file, "setp    "+pinname+".is_opendrain  true"
                            if t == _PD.SSR0:
                                temp = pinname
                            else:
                                temp = pinname + ".out"
                        # set pin true if force-pin-true otherwise connect to a signal
                        if p == "force-pin-true":
                            print >>file, "setp %s true"% (temp)
                        else:
                            print >>file, "net %s  =>     %s"% (p,temp)
                    if i: # invert pin
                        if "sserial" in pname: 
                            ending = "-invert"
                        elif "parport" in pinname: ending = "-invert"
                        else: ending = ".invert_output"
                        print >>file, "setp %s true"%  (pinname + ending )

            # for pwm pins
            elif t in (_PD.PWMP,_PD.PDMP,_PD.UDMU):
                if not p == "unused-pwm":
                    for sig in (self.d.halpwmoutputsignames):
                        if p == (sig+"-pulse"):
                            pinname = self.a.make_pinname(pname, substitution = self.d.useinisubstitution)
                            print >>file, "\n# ---",sig.upper(),"---"
                            if t == _PD.PWMP:
                                print >>file, "setp    "+pinname +".output-type 1"
                            if t == _PD.UDMU:
                                print >>file, "setp    "+pinname +".output-type 2"
                            elif t == _PD.PDMP:
                                print >>file, "setp    "+pinname +".output-type 3"
                            print >>file, "net %s     <=  "% (sig+"-enable")+pinname +".enable"
                            print >>file, "net %s      <=  "% (sig+"-value")+pinname +".value"
                            for ending in ("scale","output-type"):
                                title = sig + "-%s"% ending
                                name = pinname + ".%s"% ending
                                self._substitution_list.append((title.upper(),name))
                            self._substitution_list.append(("",""))
                            break
            # fot TP pwm pins
            elif t == (_PD.TPPWMA):
                if not p == "unused-tppwmgen":
                    for sig in (self.d.haltppwmoutputsignames):
                        if p == (sig+"-a"):
                            pinname = self.a.make_pinname(pname, substitution = self.d.useinisubstitution) 
                            print >>file, "\n# ---",sig.upper(),"---"
                            print >>file, "net %s           <=  "% (sig+"-enable")+pinname +".enable"
                            print >>file, "net %s           <=  "% (sig+"-a-value")+pinname +".A-value"
                            print >>file, "net %s           <=  "% (sig+"-b-value")+pinname +".B-value"
                            print >>file, "net %s           <=  "% (sig+"-c-value")+pinname +".C-value"
                            print >>file, "net %s           <=  "% (sig+"-fault")+pinname +".fault"
                            for ending in ("scale","deadtime","fault-invert","sample-time"):
                                title = sig + "-%s"% ending
                                name = pinname + ".%s"% ending
                                self._substitution_list.append((title.upper(),name))
                            self._substitution_list.append(("",""))
                            break
            # for stepper pins
            elif t == (_PD.STEPA):
                if not p == "unused-stepgen":
                    for sig in (self.d.halsteppersignames):
                        if p == (sig+"-step"):
                            pinname = self.a.make_pinname(pname, substitution = self.d.useinisubstitution) 
                            print >>file, "\n# ---",sig.upper(),"---"
                            print >>file, "net %s           <=  "% (sig+"-enable")+pinname +".enable"  
                            print >>file, "net %s            <=  "% (sig+"-count")+pinname +".counts" 
                            print >>file, "net %s     <=  "% (sig+"-cmd-position")+pinname +".position-cmd"  
                            print >>file, "net %s     <=  "% (sig+"-act-position")+pinname +".position-fb" 
                            print >>file, "net %s         <=  "% (sig+"-velocity")+pinname +".velocity-fb"
                            pinlist = self.a.list_related_pins([_PD.STEPA,_PD.STEPB], boardnum, connector, channel, pin, 0)
                            for i in pinlist:
                                if self.d[i[0]+"inv"]:
                                    gpioname = self.a.make_pinname(i[0],gpionumber = True, substitution = self.d.useinisubstitution)
                                    print >>file, "setp    "+gpioname+".invert_output true"
                            for ending in ("position-scale","maxvel","maxaccel","steplen","stepspace","dirsetup","dirhold","step_type"):
                                title = sig + "-%s"% ending
                                name = pinname + ".%s"% ending
                                self._substitution_list.append((title.upper(),name))
                            self._substitution_list.append(("",""))
                            break
            # potentiometer outpot
            elif t == (_PD.POTO):
                print pname
                if not p == "unused-pot":
                    for sig in (self.d.halpotsignames):
                        print "looking",p,sig
                        if p == (sig+"-output"):
                            pinname = self.a.make_pinname(pname, substitution = self.d.useinisubstitution) 
                            print >>file, "\n# ---",sig.upper(),"---"
                            print >>file, "net %s            =>  "% (sig+"-enable")+pinname +".spinena"  
                            print >>file, "net %s            =>  "% (sig+"-output")+pinname +".spinout" 
                            print >>file, "net %s          =>  "% (sig+"-direction")+pinname +".spindir"
                            pinlist = self.a.list_related_pins([_PD.POTO,_PD.POTE], boardnum, port, channel, pin, 0)
                            for i in pinlist:
                                if self.d[i[0]+"inv"]:
                                    if self.d[i[0]+"type"] == _PD.POTO:
                                        print >>file, "setp    "+pinname+".spindir-invert true"
                                    if self.d[i[0]+"type"] == _PD.POTE:
                                        print >>file, "setp    "+pinname+".spinena-invert true"
                            for ending in ("spindir-invert","spinena-invert","spinout-maxlim","spinout-minlim","spinout-scalemax"):
                                title = sig + "-%s"% ending
                                name = pinname + ".%s"% ending
                                self._substitution_list.append((title.upper(),name))
                            self._substitution_list.append(("",""))
                            break
        # mesa mainboards
        for boardnum in range(0,int(self.d.number_mesa)):
            for concount,connector in enumerate(self.d["mesa%d_currentfirmwaredata"% (boardnum)][_PD._NUMOFCNCTRS]) :
                for pin in range(0,24):
                    pname = 'mesa%dc%dpin%d' % (boardnum,connector, pin)
                    p = self.d['mesa%dc%dpin%d' % (boardnum,connector, pin)]
                    i = self.d['mesa%dc%dpin%dinv' % (boardnum,connector, pin)]
                    t = self.d['mesa%dc%dpin%dtype' % (boardnum,connector, pin)]
                    write_pins(pname,p,i,t,boardnum,connector,None,None,pin)
            # mesa sserial
            if self.d["mesa%d_numof_sserialports"% (boardnum)]: # only check if we have sserialports
                print >>file
                port = 0
                for channel in range (0,self.d["mesa%d_currentfirmwaredata"% boardnum][_PD._MAXSSERIALCHANNELS]):
                    if channel >4: break # TODO only have 5 channels worth of glade widgets
                    for pin in range (0,_PD._SSCOMBOLEN):
                        pname = 'mesa%dsserial%d_%dpin%d' % (boardnum,port,channel,pin)
                        p = self.d['mesa%dsserial%d_%dpin%d' % (boardnum,port,channel,pin)]
                        i = self.d['mesa%dsserial%d_%dpin%dinv' % (boardnum,port,channel,pin)]
                        t = self.d['mesa%dsserial%d_%dpin%dtype' % (boardnum,port,channel,pin)]
                        write_pins(pname,p,i,t,boardnum,None,port,channel,pin)
        # parports
        templist = ("pp1","pp2","pp3")
        for j, k in enumerate(templist):
            if self.d.number_pports < (j+1): break
            print >>file
            for x in (1,2,3,4,5,6,7,8,9,14,16,17):
                pname = "%s_Opin%d" % (k, x)
                p = self.d[pname]
                i = self.d[pname+"_inv"]
                if not p == "unused-output":
                    write_pins(pname,p,i,_PD.GPIOO,None,None,None,None,None)

# BOILER CODE
    def __getitem__(self, item):
        return getattr(self, item)
    def __setitem__(self, item, value):
        return setattr(self, item, value)
